<?php

namespace App\Http\Controllers\Admin;

use App\Models\System;
use App\Common\Authorizable;
use App\Events\System\DownForMaintainace;
use App\Events\System\SystemInfoUpdated;
use App\Events\System\SystemIsLive;
use App\Http\Controllers\Controller;
use App\Http\Requests\Validations\ResetDatabaseRequest;
use App\Http\Requests\Validations\SaveEnvFileRequest;
use App\Http\Requests\Validations\UpdateBasicSystemConfigRequest;
use App\Http\Requests\Validations\UpdateSystemRequest;
use App\Jobs\ResetDbAndImportDemoData;
use App\Jobs\ClearConfigCache;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Cache;
use Symfony\Component\Console\Output\BufferedOutput;

class SystemController extends Controller
{
    use Authorizable;

    private $model_name;

    /**
     * construct
     */
    public function __construct()
    {
        parent::__construct();

        $this->model_name = trans('app.model.config');
    }

    /**
     * Display the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function view()
    {
        $system = System::orderBy('id', 'asc')->first();

        return view('admin.system.general', compact('system'));
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function update(UpdateBasicSystemConfigRequest $request)
    {
        if (config('app.demo') == true) {
            return back()->with('warning', trans('messages.demo_restriction'));
        }

        $system = System::orderBy('id', 'asc')->first();

        $this->authorize('update', $system); // Check permission

        $system->update($request->except('image', 'delete_image'));

        if ($request->hasFile('icon')) {
            $request->file('icon')->storeAs('', 'icon.png');

            Storage::deleteDirectory(image_cache_path('icon.png'));
        }

        if ($request->hasFile('logo')) {
            $request->file('logo')->storeAs('', 'logo.png');

            Storage::deleteDirectory(image_cache_path('logo.png'));
        }

        if ($request->hasFile('trust_badge')) {
            $request->file('trust_badge')->storeAs('', 'trust_badge.png');

            Storage::deleteDirectory(image_cache_path('trust_badge.png'));
        }

        event(new SystemInfoUpdated($system));

        return back()->with('success', trans('messages.updated', ['model' => $this->model_name]));
    }

    /**
     * Show the .env file editor.
     *
     * @return \Illuminate\Http\Response
     */
    public function modifyEnvFile(UpdateSystemRequest $request)
    {
        $envContents = file_get_contents(base_path('.env'));

        return view('admin.system.modify_env_file', compact('envContents'));
    }

    /**
     * Reset the database and import demo data.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function saveEnvFile(SaveEnvFileRequest $request)
    {
        if (config('app.demo') == true) {
            return back()->with('warning', trans('messages.demo_restriction'));
        }

        try {
            file_put_contents(base_path('.env'), $request->env);
        } catch (\Exception $e) {
            Log::error('.env modification failed: ' . $e->getMessage());

            // add your error messages:
            $error = new \Illuminate\Support\MessageBag();
            $error->add('errors', trans('responses.failed'));

            return back()->withErrors($error);
        }

        $system = System::orderBy('id', 'asc')->first();

        event(new SystemInfoUpdated($system));

        ClearConfigCache::dispatch();

        return back()->with('success', trans('messages.env_saved'));
    }

    /**
     * Show confirmation page to import demo contents.
     *
     * @return \Illuminate\Http\Response
     */
    public function importDemoContents()
    {
        return view('admin.system.import_demo_contents');
    }

    /**
     * Reset the database and import demo data.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function resetDatabase(ResetDatabaseRequest $request)
    {
        if (config('app.demo') == true) {
            return back()->with('warning', trans('messages.demo_restriction'));
        }

        // Start transaction!
        DB::beginTransaction();

        try {
            ResetDbAndImportDemoData::dispatch();
        } catch (\Exception $e) {

            // rollback the transaction and log the error
            DB::rollback();
            Log::error('Database Reset Failed: ' . $e->getMessage());

            // add your error messages:
            $error = new \Illuminate\Support\MessageBag();
            $error->add('errors', trans('responses.failed'));

            return back()->withErrors($error);
        }

        // Everything is fine. Now commit the transaction
        DB::commit();

        // Clear the cache config
        Cache::flush();

        return back()->with('success', trans('messages.imported'));
    }

    /**
     * Toggle Maintenance Mode of the given id, Its uses the ajax middleware
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function toggleMaintenanceMode(UpdateSystemRequest $request)
    {
        if (config('app.demo') == true) {
            return response('error', 444);
        }

        $system = System::orderBy('id', 'asc')->first();

        $this->authorize('update', $system); // Check permission

        $system->maintenance_mode = !$system->maintenance_mode;

        if ($system->save()) {
            if ($system->maintenance_mode) {
                event(new DownForMaintainace($system));
            } else {
                event(new SystemIsLive($system));
            }

            // Clear the cache config
            Cache::forget('system_settings');

            return response('success', 200);
        }

        return response('error', 405);
    }

    /**
     * Take a database backup snapshot.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function backup(UpdateSystemRequest $request)
    {
        $output = '';

        try {
            $outputLog = new BufferedOutput;

            Log::info('Backup cleanup called! ');

            // Remove all backups older than specified number of days in config.
            Artisan::queue('backup:clean', [], $outputLog);

            Log::info(Artisan::output());

            Log::info('Database Backup command called!');

            Artisan::queue('backup:run', ['--only-db' => true], $outputLog);

            Log::info(Artisan::output());
        } catch (Exception $e) {
            Log::error('Backup failed! ' . $outputLog);

            return back()->withErrors('Backup failed: ' . $output);
        }

        return back()->with('success', trans('messages.backup_done'));
    }

    /**
     * Uninstall the application license so that it can be reinstall on new location.
     * Script stops working immediately.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function uninstallAppLicense(UpdateSystemRequest $request)
    {
        if ($request->isMethod('get')) {
            return view('admin.system.uninstall');
        }

        if ($request->do_action != 'UNINSTALL') {
            return back()->withErrors(trans('validation.do_action_invalid'));
        }

        if (Hash::check($request->password, $request->user()->password)) {
            // Start transaction!
            DB::beginTransaction();

            try {
                $license_notifications_array = incevioUninstallLicense(getMysqliConnection());
            } catch (\Exception $e) {
                // rollback the transaction and log the error
                DB::rollback();

                Log::error('License uninstallation failed: ' . $license_notifications_array['notification_text']);

                // add your error messages:
                $error = new \Illuminate\Support\MessageBag();
                $error->add('errors', trans('responses.failed'));

                return back()->withErrors($error);
            }

            if ($license_notifications_array['notification_case'] == 'notification_license_ok') {
                // Everything is fine. Now commit the transaction
                DB::commit();

                // Delete the installed file
                unlink(storage_path('installed'));

                // $MYSQLI_LINK = getMysqliConnection();
                // mysqli_query($MYSQLI_LINK, "SET FOREIGN_KEY_CHECKS = 0");
                // mysqli_query($MYSQLI_LINK, "DROP TABLE ".APL_DATABASE_TABLE);

                return back()->with('success', trans('messages.license_uninstalled'));
            }

            // rollback the transaction and log the error
            DB::rollback();

            Log::error('License uninstallation failed: ' . $license_notifications_array['notification_text']);

            return back()->withErrors('License uninstallation failed: ' . $license_notifications_array['notification_text']);
        }

        abort(403, 'Unauthorized action.');
    }

    /**
     * Update the application license if the IP has been changed.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function updateAppLicense(UpdateSystemRequest $request)
    {
        $license_notifications_array = incevioUpdateLicense(getMysqliConnection());

        if ($license_notifications_array['notification_case'] == 'notification_license_ok') {
            return back()->with('success', trans('messages.license_updated'));
        }

        Log::error('License update failed: ' . $license_notifications_array['notification_text']);

        return back()->withErrors('License update failed: ' . $license_notifications_array['notification_text']);
    }
}
